// Frontend JS for simple UI interactions
async function api(path, opts={}) {
  const res = await fetch('/api' + path, opts);
  if (!res.ok) {
    const t = await res.text().catch(()=>null);
    throw new Error(t || res.statusText);
  }
  return res.json();
}

// Dashboard
async function loadDashboard() {
  try {
    const students = await api('/students');
    document.getElementById('total-students').textContent = 'Siswa: ' + students.length;

    const feesAll = await fetch('/api/fees').catch(()=>null); // not used; keep simple
    const payments = await api('/payments');
    document.getElementById('total-paid').textContent = 'Terkumpul: ' + payments.reduce((s,p)=>s+Number(p.amount),0);

    const paymentsList = document.getElementById('payments-list');
    if (paymentsList) {
      paymentsList.innerHTML = payments.length ? payments.map(p=>`<div class="card mb-2 p-2"><b>${p.name}</b> — ${p.month} ${p.year} — Rp ${p.amount} <small class="text-muted">(${p.paid_at})</small></div>`).join('') : '<div class="text-muted">Belum ada pembayaran</div>';
    }
  } catch (err) {
    console.error(err);
  }
}

// Students page
async function loadStudentsPage() {
  const tbl = document.querySelector('#students-table tbody');
  const stuSelect = document.getElementById('fee-student');
  const students = await api('/students');
  if (tbl) {
    tbl.innerHTML = students.map((s,i)=>`<tr>
      <td>${i+1}</td><td>${s.name}</td><td>${s.class || ''}</td><td>${s.parent_name || ''}</td>
      <td>
        <button class="btn btn-sm btn-outline-secondary" onclick="viewFees(${s.id})">Tagihan</button>
        <button class="btn btn-sm btn-danger" onclick="deleteStudent(${s.id})">Hapus</button>
      </td>
    </tr>`).join('');
  }
  if (stuSelect) {
    stuSelect.innerHTML = `<option value="">Pilih siswa</option>` + students.map(s=>`<option value="${s.id}">${s.name} (${s.class||''})</option>`).join('');
  }
}

async function addStudent() {
  const name = document.getElementById('name').value.trim();
  const cls = document.getElementById('class').value.trim();
  const parent = document.getElementById('parent').value.trim();
  try {
    await api('/students', { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify({ name, class: cls, parent_name: parent })});
    document.getElementById('name').value=''; document.getElementById('class').value=''; document.getElementById('parent').value='';
    await loadStudentsPage();
  } catch (err) { document.getElementById('stu-msg').textContent = err.message; }
}

async function deleteStudent(id) {
  if (!confirm('Hapus siswa dan semua data terkait?')) return;
  try {
    await api('/students/' + id, { method: 'DELETE' });
    await loadStudentsPage();
  } catch (err) { alert(err.message); }
}

// Fees page
async function createFee() {
  const student_id = document.getElementById('fee-student').value;
  const month = document.getElementById('fee-month').value.trim();
  const year = Number(document.getElementById('fee-year').value);
  const amount = Number(document.getElementById('fee-amount').value);
  if (!student_id || !month || !year || !amount) { document.getElementById('fee-msg').textContent = 'Lengkapi semua field'; return; }
  try {
    await api('/fees', { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify({ student_id, month, year, amount })});
    document.getElementById('fee-month').value=''; document.getElementById('fee-amount').value='';
    await viewFees(student_id);
  } catch (err) { document.getElementById('fee-msg').textContent = err.message; }
}

async function viewFees(student_id) {
  const el = document.getElementById('fees-area');
  try {
    const fees = await api('/fees/student/' + student_id);
    if (!el) return;
    el.innerHTML = fees.length ? fees.map(f=>`
      <div class="card mb-2 p-2">
        <b>${f.name}</b> — ${f.month} ${f.year} — Rp ${f.amount} — <span class="badge bg-${f.status==='paid'?'success':'warning'}">${f.status}</span>
        ${f.status!=='paid' ? `<div class="mt-2"><button class="btn btn-sm btn-primary" onclick="payFee(${f.id}, ${f.amount})">Tandai Lunas (Bayar Rp ${f.amount})</button></div>` : ''}
      </div>
    `).join('') : '<div class="text-muted">Belum ada tagihan</div>';
  } catch (err) { el.innerHTML = '<div class="text-danger">Gagal memuat</div>'; console.error(err); }
}

async function payFee(fee_id, amount) {
  if (!confirm('Konfirmasi pembayaran?')) return;
  try {
    await api('/payments/pay', { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify({ fee_id, amount })});
    alert('Pembayaran dicatat');
    // reload lists
    await loadStudentsPage();
    const sel = document.getElementById('fee-student');
    if (sel && sel.value) await viewFees(sel.value);
    await loadDashboard();
  } catch (err) { alert(err.message); }
}

// Wire buttons if present
document.addEventListener('DOMContentLoaded', () => {
  if (document.querySelector('#add-student')) {
    document.getElementById('add-student').addEventListener('click', addStudent);
    loadStudentsPage();
  }
  if (document.getElementById('create-fee')) {
    document.getElementById('create-fee').addEventListener('click', createFee);
    loadStudentsPage(); // fills student select
  }
  loadDashboard();
});