const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const fs = require('fs');

const DB_FILE = process.env.DATABASE_FILE || path.join(__dirname, 'data', 'syahriah.db');

// ensure data folder exists
const dataDir = path.dirname(DB_FILE);
if (!fs.existsSync(dataDir)) fs.mkdirSync(dataDir, { recursive: true });

const db = new sqlite3.Database(DB_FILE);

// Initialize tables
db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS students (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT NOT NULL,
      class TEXT,
      parent_name TEXT,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS fees (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      student_id INTEGER NOT NULL,
      month TEXT NOT NULL,
      year INTEGER NOT NULL,
      amount INTEGER NOT NULL,
      status TEXT NOT NULL DEFAULT 'unpaid',
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY(student_id) REFERENCES students(id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS payments (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      fee_id INTEGER,
      student_id INTEGER,
      amount INTEGER NOT NULL,
      paid_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      note TEXT,
      FOREIGN KEY(fee_id) REFERENCES fees(id),
      FOREIGN KEY(student_id) REFERENCES students(id)
    )
  `);
});

// Helper DB functions (promisified)
const run = (sql, params=[]) => new Promise((resolve, reject) =>
  db.run(sql, params, function(err) {
    if (err) reject(err); else resolve(this);
  })
);
const get = (sql, params=[]) => new Promise((resolve, reject) =>
  db.get(sql, params, (err, row) => err ? reject(err) : resolve(row))
);
const all = (sql, params=[]) => new Promise((resolve, reject) =>
  db.all(sql, params, (err, rows) => err ? reject(err) : resolve(rows))
);

// Exported functions
module.exports = {
  run, get, all,
  // convenience wrappers
  async createStudent(name, cls, parent_name) {
    const r = await run(`INSERT INTO students (name, class, parent_name) VALUES (?, ?, ?)`, [name, cls, parent_name]);
    return r.lastID;
  },
  async listStudents() {
    return all(`SELECT * FROM students ORDER BY name`);
  },
  async getStudent(id) {
    return get(`SELECT * FROM students WHERE id = ?`, [id]);
  },
  async updateStudent(id, name, cls, parent_name) {
    await run(`UPDATE students SET name=?, class=?, parent_name=? WHERE id=?`, [name, cls, parent_name, id]);
    return this.getStudent(id);
  },
  async deleteStudent(id) {
    await run(`DELETE FROM students WHERE id=?`, [id]);
    await run(`DELETE FROM fees WHERE student_id=?`, [id]);
    await run(`DELETE FROM payments WHERE student_id=?`, [id]);
    return true;
  },

  async createFee(student_id, month, year, amount) {
    const r = await run(`INSERT INTO fees (student_id, month, year, amount) VALUES (?, ?, ?, ?)`, [student_id, month, year, amount]);
    return r.lastID;
  },
  async listFeesByStudent(student_id) {
    return all(`SELECT f.*, s.name FROM fees f LEFT JOIN students s ON s.id=f.student_id WHERE f.student_id=? ORDER BY year DESC, month DESC`, [student_id]);
  },
  async getFee(id) {
    return get(`SELECT f.*, s.name FROM fees f LEFT JOIN students s ON s.id=f.student_id WHERE f.id=?`, [id]);
  },
  async markFeePaid(fee_id, amount, note='') {
    const fee = await get(`SELECT * FROM fees WHERE id=?`, [fee_id]);
    if (!fee) throw new Error('Fee not found');
    await run(`INSERT INTO payments (fee_id, student_id, amount, note) VALUES (?, ?, ?, ?)`, [fee_id, fee.student_id, amount, note]);
    await run(`UPDATE fees SET status='paid' WHERE id=?`, [fee_id]);
  },

  async listPayments() {
    return all(`SELECT p.*, s.name, f.month, f.year FROM payments p LEFT JOIN students s ON s.id=p.student_id LEFT JOIN fees f ON f.id=p.fee_id ORDER BY p.paid_at DESC`);
  },

  async allPayments() {
    return all(`SELECT p.*, s.name, f.month, f.year FROM payments p LEFT JOIN students s ON s.id=p.student_id LEFT JOIN fees f ON f.id=p.fee_id ORDER BY p.paid_at DESC`);
  }
};